#!/bin/bash
# vim: tabstop=4 shiftwidth=4 softtabstop=4 expandtab

INSTALL_BUILD_DEPS=0
INSTALL_BUILD_DEPS_ONLY=0
PURGE_SOURCE_DIRECTORY=0
SID=0

function usage() {
    [[ $# -gt 0 ]] && echo -e "$*\n" > /dev/stderr
    cat << EOF
Usage: $0 [-h|I|-O|-P|-x|--sid]
    -h|--help                       Show this message
    -I|--install-build-deps         Install build dependencies before building package
    -O|--install-build-deps-only    Only install build dependencies
    -P|--purge-sources-directory    Purge debian package sources directory after building package
    -x|--trace                      Enable bash tracing (set -x)
    --sid                           Build package for Debian Sid
EOF
    [[ $# -gt 0 ]] && exit 1
} 

idx=1
while [[ $idx -le $# ]]; do
    OPT=${!idx}
    case $OPT in
        -h|--help)
            usage;;
        -I|--install-build-deps)
            INSTALL_BUILD_DEPS=1;;
        -O|--install-build-deps-only)
            INSTALL_BUILD_DEPS=1
            INSTALL_BUILD_DEPS_ONLY=1
            ;;
        -P|--purge-sources-directory)
            PURGE_SOURCE_DIRECTORY=1;;
        --sid)
            SID=1;;
        -x|--trace)
            set -x;;
        *)
            usage "Unknown parameter '$OPT'"
    esac
    (( idx+=1 ))
done

# Enter source directory
cd "$( dirname "$0" )"

# Install build dependencies
if [[ $INSTALL_BUILD_DEPS -eq 1 ]]; then
  apt-get update
  apt-get install --no-install-recommends --yes \
    devscripts equivs sed rsync git lsb-release wget ca-certificates
  mk-build-deps --install --tool='apt-get -o Debug::pkgProblemResolver=yes --no-install-recommends --yes' debian/control
fi

DEBIAN_RELEASE=$( lsb_release -r -s | sed 's/\..*$//' )

# Install build dependencies based on Debian release
if [[ $INSTALL_BUILD_DEPS -eq 1 ]]; then
  # On Debian Stretch, install GitPython using pip because strect version of python3-git have bugs
  if [[ $DEBIAN_RELEASE -eq 9 ]]; then
    apt-get install --no-install-recommends --yes python3-pip
    python3 -m pip install GitPython
  else
    apt-get install --no-install-recommends --yes python3-git
  fi
fi

# Install GPG key (if provided)
if [[ -n "$GPG_KEY" ]]; then
    [[ $INSTALL_BUILD_DEPS -eq 1 ]] && apt-get install --no-install-recommends --yes gnupg2
    [[ $INSTALL_BUILD_DEPS_ONLY -eq 0 ]] && base64 -d <<< "$GPG_KEY" | gpg --import
fi

# Stop here on install build deps only mode
[[ $INSTALL_BUILD_DEPS_ONLY -eq 1 ]] && exit 0

set -e

# Retreive source package name
SOURCE_PACKAGE_NAME=$( grep -E ^Source: debian/control|sed 's/^Source: //' )

if [[ $SID -eq 1 ]]; then
  DEBIAN_CODENAME=sid
else
  DEBIAN_CODENAME=$( lsb_release -c -s )
fi

# Clean previous build
rm -fr "$SOURCE_PACKAGE_NAME-build-deps"* dist

# Compute version using git describe
# Note : If no tag exist, git describe will fail: in this case, compute a 0.0 version with same
# format as git describe
VERSION="$( git describe --tags 2> /dev/null )" || \
    VERSION="0.0-$( git log --oneline | wc -l )-$( git describe --tags --always )"

# Create dist directory
DIST_DIR="dist/$SOURCE_PACKAGE_NAME-$VERSION"
mkdir -p "$DIST_DIR"
rsync -av \
  --exclude 'local/*' \
  --exclude 'local.*' \
  --exclude 'tmp/*' \
  --exclude '.placeholder' \
  --exclude '.gitignore' \
  --exclude '*~' \
  ./src/ "$DIST_DIR/src/"
rsync -av \
  --exclude 'files' \
  --exclude 'ldapsaisie.debhelper.log' \
  --exclude 'ldapsaisie.substvars' \
  --exclude 'ldapsaisie' \
  --exclude 'debhelper-build-stamp' \
  --exclude '.debhelper' \
  --exclude '.placeholder' \
  --exclude '.gitignore' \
  --exclude '*~' \
  ./debian/ "$DIST_DIR/debian/"
rsync -av \
  --exclude 'public_html' \
  --exclude '*.html' \
  --exclude '*.pdf' \
  --exclude '*.epub' \
  --exclude 'venv' \
  --exclude 'dist' \
  --exclude '.placeholder' \
  --exclude '.gitignore' \
  --exclude '*~' \
  ./doc/ "$DIST_DIR/doc/"
rsync -av \
  --exclude '.placeholder' \
  --exclude '.gitignore' \
  --exclude '*~' \
  ./lsexample/ "$DIST_DIR/lsexample/"

# Check gitdch is installed
GITDCH=$(which gitdch)
set -e
if [[ -z "$GITDCH" ]]; then
  TMP_GITDCH=$(mktemp -d)
  echo "Temporary install gitdch in $TMP_GITDCH"
  wget -O "$TMP_GITDCH/gitdch" https://gitea.zionetrix.net/bn8/gitdch/raw/master/gitdch
  chmod +x "$TMP_GITDCH/gitdch"
  GITDCH=$TMP_GITDCH/gitdch
else
  TMP_GITDCH=""
fi

# Compute gitdch extra args
GITDCH_EXTRA_ARGS=()
[[ -n "$DEBFULLNAME" ]] && GITDCH_EXTRA_ARGS+=( "--maintainer-name" "$DEBFULLNAME" )
[[ -n "$DEBEMAIL" ]] && GITDCH_EXTRA_ARGS+=( "--maintainer-email" "$DEBEMAIL" )

if [[ -e debian/changelog ]]; then
    LAST_MANUAL_COMMIT_ID=$( git log --oneline -n1 -- debian/changelog | awk '{print $1}' )
    GITDCH_EXTRA_ARGS+=( "--append" "--revision" "${LAST_MANUAL_COMMIT_ID}..HEAD" )
fi

# Generate debian changelog using generate_debian_changelog.py
$GITDCH \
  --package-name "$SOURCE_PACKAGE_NAME" \
  --version="${VERSION}" \
  --code-name "$DEBIAN_CODENAME" \
  --output "$DIST_DIR/debian/changelog" \
  --release-notes dist/release-notes.md \
  --exclude "^Docker: " \
  --exclude "^CI: " \
  --exclude "^debian: " \
  --exclude "\.gitlab-ci(\.yml)?" \
  --exclude "build-deb(\.sh)?" \
  --exclude "README(\.md)?" \
  --exclude "^Merge branch " \
  --verbose "${GITDCH_EXTRA_ARGS[@]}"

# Clean temporary gitdch installation
[[ -n "$TMP_GITDCH" ]] && rm -fr "$TMP_GITDCH"

# Build debian package
BUILD_ARGS=()
if [[ -z "$GPG_KEY" ]]; then
    BUILD_ARGS+=("--no-sign")
else
    echo "GPG key provide, enable package signing."
fi
cd "$DIST_DIR/"
if dpkg-buildpackage "${BUILD_ARGS[@]}"; then
    # Handle PURGE_SOURCE_DIRECTORY option
    [[ $PURGE_SOURCE_DIRECTORY -eq 0 ]] || rm -fr "../../$DIST_DIR"
else
    echo "Failed to build debian package"
    exit 1
fi
